package ga.view.processor;

import ga.view.factory.EffectsFactory;

import java.util.logging.Logger;

import com.jme3.asset.AssetManager;
import com.jme3.font.BitmapFont;
import com.jme3.font.BitmapText;
import com.jme3.material.Material;
import com.jme3.material.RenderState.BlendMode;
import com.jme3.math.ColorRGBA;
import com.jme3.math.Vector3f;
import com.jme3.post.SceneProcessor;
import com.jme3.renderer.Camera;
import com.jme3.renderer.RenderManager;
import com.jme3.renderer.ViewPort;
import com.jme3.renderer.queue.RenderQueue;
import com.jme3.renderer.queue.RenderQueue.Bucket;
import com.jme3.scene.Node;
import com.jme3.scene.Spatial.CullHint;
import com.jme3.system.AppSettings;
import com.jme3.texture.FrameBuffer;
import com.jme3.texture.Image.Format;
import com.jme3.texture.Texture2D;

/**
 * Scene processor for offscreen rendering of textures. Allows to render a scene
 * onto a texture.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
public class OffscreenProcessor implements SceneProcessor {

  // the logger for this class
  private static final Logger LOGGER = Logger
      .getLogger(OffscreenProcessor.class.getName());

  private int renderWidth = 512;
  private int renderHeight = 512;
  private FrameBuffer offscreenBuffer;
  private Camera offscreenCam;
  private final Node offscreenScene;
  private ViewPort offscreenView;
  private final AssetManager assetManager;
  private RenderManager rm;
  private final AppSettings settings;

  // /
  private ViewPort guiViewPort;
  private Node guiNode;
  private BitmapText fpsText;
  private BitmapFont guiFont;
  private String text = "";
  //

  private ColorRGBA backgroundColor = ColorRGBA.White;

  // the texture to render to
  private Texture2D texture;

  // the material that will be created
  private final Material material;

  // the key to apply the texture to the material
  private final String textureKey;

  /**
   * Creates a new offscreen texture processor with a SimpleTextured material.
   * 
   * @param assetManager
   *          The AssetManager
   * @param settings
   *          The application settings
   * @param scene
   *          The scene that will be rendered
   * @param material
   *          The material to apply the texture to
   * @param textureKey
   *          The texture key used to apply the texture
   */
  public OffscreenProcessor(final AssetManager assetManager,
      final AppSettings settings, final Node scene, final Material material,
      final String textureKey) {
    this.assetManager = assetManager;
    this.settings = settings;
    this.offscreenScene = scene;
    this.material = material;
    this.textureKey = textureKey;

    init();
  }

  /**
   * Pre-initializes the processor.
   */
  private void init() {
    texture = new Texture2D(renderWidth, renderHeight, Format.RGBA8);

    offscreenCam = new Camera(renderWidth, renderHeight);
  }

  /**
   * Sets the size of the offscreen texture.
   * 
   * @param width
   *          With of the texture
   * @param height
   *          Height of the texture
   */
  public void setTextureSize(final int width, final int height) {
    this.renderWidth = width;
    this.renderHeight = height;

    init();
  }

  @Override
  public void initialize(final RenderManager rm, final ViewPort vp) {
    this.rm = rm;

    // create a pre-view. a view that is rendered before the main view
    offscreenView = rm.createPreView("Offscreen View", offscreenCam);
    // offscreenCam.setFrustum(vp.getCamera().getFrustumNear(), vp.getCamera()
    // .getFrustumFar(), vp.getCamera().getFrustumLeft(), vp.getCamera()
    // .getFrustumRight(), vp.getCamera().getFrustumTop(), vp.getCamera()
    // .getFrustumBottom());

    offscreenView.setClearFlags(true, true, true);
    offscreenView.setBackgroundColor(backgroundColor);
    // create offscreen framebuffer
    offscreenBuffer = new FrameBuffer(renderWidth, renderHeight, 1);
    // setup framebuffer to use texture
    offscreenBuffer.setDepthBuffer(Format.Depth);
    offscreenBuffer.setColorTexture(texture);

    // set viewport to render to offscreen framebuffer
    offscreenView.setOutputFrameBuffer(offscreenBuffer);

    // attach the scene to the viewport to be rendered
    offscreenView.attachScene(offscreenScene);

    // apply the texture
    material.setTexture(textureKey, texture);
    material.setColor("Diffuse", backgroundColor);

    final float aspect = settings.getHeight() / (float) settings.getWidth();

    // BEGIN this has been added for osd
    // Create a new cam for the gui
    // final Camera guiCam = new Camera(renderWidth, renderHeight);
    guiViewPort = rm.createPreView("Gui Offscreen", offscreenCam);
    guiViewPort.setClearFlags(false, false, false);

    guiNode = new Node("Gui Node");
    guiViewPort.attachScene(guiNode);

    guiNode.setQueueBucket(Bucket.Gui);
    guiNode.setCullHint(CullHint.Never);

    guiFont = assetManager.loadFont("Interface/Fonts/Default.fnt");
    fpsText = new BitmapText(guiFont, false);
    fpsText.setSize(64);
    fpsText.setLocalTranslation(2f, fpsText.getLineHeight() * 1.5f, 0);
    fpsText.setLocalScale(aspect, 1f, 1f);
    fpsText.setText(text);
    fpsText.setColor(ColorRGBA.Red);
    guiNode.attachChild(fpsText);

    guiViewPort.setOutputFrameBuffer(offscreenBuffer);

    material.getAdditionalRenderState().setBlendMode(BlendMode.Alpha);
    // // END

    EffectsFactory.addShadowProcessor(assetManager, settings, offscreenView,
        new Vector3f(300f, 210f, 300f));
    EffectsFactory.addLightScatteringProcessor(assetManager, null, settings,
        offscreenView, new Vector3f(300f, 190f, 300f));
    EffectsFactory
        .addSSAOProcessor(assetManager, null, settings, offscreenView);

    // update the state of the offscreen scene
    offscreenScene.updateGeometricState();
  }

  /**
   * Sets a text that is displayed on the GUI.
   * 
   * @param text
   *          GUI text.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void setText(final String text) {
    this.text = text;
    if (fpsText != null) {
      fpsText.setText(text);
    }
  }

  @Override
  public void reshape(final ViewPort vp, final int w, final int h) {
  }

  @Override
  public boolean isInitialized() {
    return rm != null;
  }

  @Override
  public void preFrame(final float tpf) {
    guiNode.updateLogicalState(tpf);
    guiNode.updateGeometricState();
  }

  @Override
  public void postQueue(final RenderQueue rq) {
  }

  @Override
  public void postFrame(final FrameBuffer out) {
  }

  @Override
  public void cleanup() {
    try {
      rm.removePreView(offscreenView);
      rm.removePreView(guiViewPort);
    } catch (final Throwable t) {
      LOGGER.warning(t.toString());
    }
  }

  /**
   * Getter for the camera.
   * 
   * @return The camera of the offscreen scene
   */
  public Camera getCamera() {
    return offscreenCam;
  }

  /**
   * Getter for the material to render to.
   * 
   * @return The material on which is rendered
   */
  public Material getMaterial() {
    return material;
  }

  /**
   * Getter for the background color of the scene.
   * 
   * @return The color of the background
   */
  public ColorRGBA getBackgroundColor() {
    return backgroundColor;
  }

  /**
   * Setter for the background color of the scene.
   * 
   * @param backgroundColor
   *          The color of the background
   */
  public void setBackgroundColor(final ColorRGBA backgroundColor) {
    this.backgroundColor = backgroundColor;

    if (isInitialized()) {
      offscreenView.setBackgroundColor(backgroundColor);

      material.setColor("Diffuse", backgroundColor);
    }
  }

}
